"use client"

import { createContext, useContext, useState, useEffect, type ReactNode } from "react"
import type { User } from "./types"

interface AuthContextType {
  user: User | null
  login: (username: string, password: string) => { success: boolean; error?: string }
  register: (username: string, password: string, displayName: string) => { success: boolean; error?: string }
  logout: () => void
  updateProfile: (updates: Partial<Pick<User, "displayName" | "status">>) => void
  isLoading: boolean
}

const AuthContext = createContext<AuthContextType | undefined>(undefined)

export function AuthProvider({ children }: { children: ReactNode }) {
  const [user, setUser] = useState<User | null>(null)
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    const storedUser = localStorage.getItem("currentUser")
    if (storedUser) {
      setUser(JSON.parse(storedUser))
    }
    setIsLoading(false)
  }, [])

  const getUsers = (): User[] => {
    const users = localStorage.getItem("users")
    return users ? JSON.parse(users) : []
  }

  const saveUsers = (users: User[]) => {
    localStorage.setItem("users", JSON.stringify(users))
  }

  const register = (username: string, password: string, displayName: string) => {
    const users = getUsers()

    if (users.find((u) => u.username === username)) {
      return { success: false, error: "Bu kullanıcı adı zaten kullanılıyor" }
    }

    const newUser: User = {
      id: crypto.randomUUID(),
      username,
      password,
      displayName,
      status: "Merhaba! WhatsApp kullanıyorum.",
      lastSeen: new Date(),
    }

    users.push(newUser)
    saveUsers(users)

    localStorage.setItem("currentUser", JSON.stringify(newUser))
    setUser(newUser)

    return { success: true }
  }

  const login = (username: string, password: string) => {
    const users = getUsers()
    const foundUser = users.find((u) => u.username === username && u.password === password)

    if (!foundUser) {
      return { success: false, error: "Kullanıcı adı veya şifre hatalı" }
    }

    localStorage.setItem("currentUser", JSON.stringify(foundUser))
    setUser(foundUser)
    return { success: true }
  }

  const logout = () => {
    localStorage.removeItem("currentUser")
    setUser(null)
  }

  const updateProfile = (updates: Partial<Pick<User, "displayName" | "status">>) => {
    if (!user) return

    const updatedUser = { ...user, ...updates }
    setUser(updatedUser)
    localStorage.setItem("currentUser", JSON.stringify(updatedUser))

    // Update in users list too
    const users = getUsers()
    const userIndex = users.findIndex((u) => u.id === user.id)
    if (userIndex !== -1) {
      users[userIndex] = updatedUser
      saveUsers(users)
    }
  }

  return (
    <AuthContext.Provider value={{ user, login, register, logout, updateProfile, isLoading }}>
      {children}
    </AuthContext.Provider>
  )
}

export function useAuth() {
  const context = useContext(AuthContext)
  if (context === undefined) {
    throw new Error("useAuth must be used within an AuthProvider")
  }
  return context
}
