"use client"

import type React from "react"
import { useState, useRef, useEffect } from "react"
import { useAuth } from "@/lib/auth-context"
import { useChat } from "@/lib/chat-context"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { MoreVertical, Search, Paperclip, Mic, Send, Check, CheckCheck, ArrowLeft } from "lucide-react"
import { format } from "date-fns"
import { tr } from "date-fns/locale"
import { EmojiPicker } from "./emoji-picker"
import { MessageSearch } from "./message-search"

export function ChatWindow() {
  const { user } = useAuth()
  const { activeChat, messages, sendMessage, getContactInfo, setActiveChat } = useChat()
  const [newMessage, setNewMessage] = useState("")
  const messagesEndRef = useRef<HTMLDivElement>(null)
  const [showSearch, setShowSearch] = useState(false)
  const [highlightedMessageId, setHighlightedMessageId] = useState<string | null>(null)
  const messageRefs = useRef<Record<string, HTMLDivElement | null>>({})

  const contact = activeChat ? getContactInfo(activeChat.participants) : null
  const chatMessages = messages.filter((m) => m.chatId === activeChat?.id)

  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" })
  }, [chatMessages])

  useEffect(() => {
    if (highlightedMessageId) {
      const timer = setTimeout(() => setHighlightedMessageId(null), 2000)
      return () => clearTimeout(timer)
    }
  }, [highlightedMessageId])

  const handleSend = () => {
    if (newMessage.trim()) {
      sendMessage(newMessage)
      setNewMessage("")
    }
  }

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault()
      handleSend()
    }
  }

  const handleEmojiSelect = (emoji: string) => {
    setNewMessage((prev) => prev + emoji)
  }

  const handleNavigateToMessage = (messageId: string) => {
    setHighlightedMessageId(messageId)
    messageRefs.current[messageId]?.scrollIntoView({ behavior: "smooth", block: "center" })
  }

  if (!activeChat || !contact) {
    return (
      <div className="flex-1 hidden md:flex flex-col items-center justify-center bg-background">
        <div className="text-center p-8">
          <div className="w-64 h-64 mx-auto mb-6 opacity-30">
            <svg viewBox="0 0 303 172" className="w-full h-full fill-muted-foreground">
              <path d="M229.565 160.229c-1.167.675-2.453 1.119-3.801 1.311h-.012c-7.227 1.033-14.52-3.051-16.547-9.675a17.11 17.11 0 01-.122-.591c-.243-.931-.393-1.907-.453-2.917-.092-1.552-.057-3.17.103-4.851l2.925-30.851h-.001c.012-.134.03-.273.043-.408l.023-.238.011-.12c.005-.078.015-.152.025-.228.07-.609.154-1.208.249-1.797.115-.75.25-1.487.402-2.211a43.478 43.478 0 011.72-5.884c1.87-5.203 4.611-9.916 8.082-13.967 1.74-2.03 3.677-3.881 5.794-5.521a41.97 41.97 0 017.097-4.597 42.291 42.291 0 0116.05-5.198 42.485 42.485 0 016.447-.494h.164c3.164 0 6.398.295 9.501.914l.131.026a42.548 42.548 0 0117.01 7.165c4.938 3.638 9.04 8.348 12.115 13.822a37.87 37.87 0 011.968 4.01c.402.972.77 1.966 1.104 2.98.15.456.293.916.427 1.378a39.66 39.66 0 011.407 6.844c.18 1.465.292 2.962.337 4.487.001.021.001.041.001.061l.001.081-.002.079c.007.394.012.789.012 1.186 0 .216-.003.432-.009.647-.003.155-.004.309-.009.463l-.003.116-.012.321-.011.292-.003.108c-.001.036-.004.072-.006.108-.018.335-.042.669-.069 1.001a42.374 42.374 0 01-2.697 11.235c-.195.495-.401.985-.617 1.471a42.376 42.376 0 01-5.122 8.633 42.32 42.32 0 01-7.435 7.189c-.154.116-.309.231-.466.346a42.274 42.274 0 01-10.598 5.656 42.232 42.232 0 01-15.583 2.929 42.316 42.316 0 01-6.257-.467l-.076-.01a42.357 42.357 0 01-4.226-.774h-.001l-20.214 11.674a.29.29 0 01-.145.039.29.29 0 01-.113-.022.287.287 0 01-.117-.077.287.287 0 01-.063-.254l3.177-17.073-.006-.003z" />
              <path
                fill="#fff"
                fillOpacity=".08"
                d="M268.3 93.371c-.157-1.01-.378-2.058-.667-3.14a35.623 35.623 0 00-1.019-3.014c-.394-.99-.85-1.943-1.364-2.852a29.77 29.77 0 00-8.461-9.766c-3.654-2.751-7.88-4.63-12.413-5.51l-.083-.016a32.876 32.876 0 00-6.153-.588h-.106c-1.437 0-2.895.107-4.323.318a32.73 32.73 0 00-12.387 4.02 32.498 32.498 0 00-5.49 3.562 32.546 32.546 0 00-4.479 4.272c-2.687 3.139-4.806 6.789-6.261 10.8a33.593 33.593 0 00-1.33 4.549c-.118.561-.22 1.131-.311 1.71a33.67 33.67 0 00-.193 1.39l-.009.098-.018.182c-.01.113-.024.227-.035.341l-2.265 23.882c-.123 1.3-.148 2.529-.075 3.69.045.772.149 1.517.309 2.231.013.06.027.117.041.176.06.253.128.498.203.739 1.576 5.138 6.852 8.301 12.167 7.542h.007c1.041-.148 2.035-.458 2.938-.907l19.836-11.455.123.018c1.169.168 2.357.285 3.548.352a33.32 33.32 0 005.188-.007c4.53-.391 8.887-1.697 12.801-3.79.269-.143.535-.292.798-.443a33.008 33.008 0 008.189-6.357 32.818 32.818 0 005.765-7.87 32.892 32.892 0 002.586-8.716c.037-.26.069-.522.099-.785a36.092 36.092 0 00.063-5.428z"
              />
            </svg>
          </div>
          <h2 className="text-2xl font-light text-foreground mb-2">WhatsApp Web</h2>
          <p className="text-muted-foreground max-w-md">
            Sohbet başlatmak için sol taraftaki listeden bir kişi seçin veya yeni sohbet başlatın.
          </p>
        </div>
      </div>
    )
  }

  return (
    <div className="flex-1 flex flex-col bg-background relative">
      <div
        className="absolute inset-0 opacity-5"
        style={{
          backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fillRule='evenodd'%3E%3Cg fill='%23ffffff' fillOpacity='1'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`,
        }}
      />

      {showSearch ? (
        <MessageSearch
          messages={chatMessages}
          onClose={() => setShowSearch(false)}
          onNavigate={handleNavigateToMessage}
        />
      ) : (
        <div className="h-16 px-4 flex items-center justify-between bg-secondary relative z-10">
          <div className="flex items-center gap-3">
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setActiveChat(null)}
              className="text-foreground hover:bg-popover md:hidden"
            >
              <ArrowLeft className="w-5 h-5" />
            </Button>
            <Avatar className="w-10 h-10">
              <AvatarFallback className="bg-accent text-foreground">
                {contact.displayName?.charAt(0).toUpperCase()}
              </AvatarFallback>
            </Avatar>
            <div>
              <span className="font-medium text-foreground block">{contact.displayName}</span>
              <span className="text-xs text-muted-foreground">{contact.status}</span>
            </div>
          </div>
          <div className="flex items-center gap-2">
            <Button
              variant="ghost"
              size="icon"
              className="text-muted-foreground hover:text-foreground hover:bg-popover"
              onClick={() => setShowSearch(true)}
            >
              <Search className="w-5 h-5" />
            </Button>
            <Button
              variant="ghost"
              size="icon"
              className="text-muted-foreground hover:text-foreground hover:bg-popover"
            >
              <MoreVertical className="w-5 h-5" />
            </Button>
          </div>
        </div>
      )}

      <div className="flex-1 overflow-y-auto p-4 relative z-10">
        {chatMessages.map((message) => {
          const isOwn = message.senderId === user?.id
          const isHighlighted = message.id === highlightedMessageId
          return (
            <div
              key={message.id}
              ref={(el) => {
                messageRefs.current[message.id] = el
              }}
              className={`flex mb-2 ${isOwn ? "justify-end" : "justify-start"}`}
            >
              <div
                className={`max-w-[65%] rounded-lg px-3 py-2 transition-all duration-300 ${
                  isOwn
                    ? "bg-chat-outgoing text-foreground rounded-br-none"
                    : "bg-chat-incoming text-foreground rounded-bl-none"
                } ${isHighlighted ? "ring-2 ring-primary ring-offset-2 ring-offset-background" : ""}`}
              >
                <p className="text-sm break-words">{message.content}</p>
                <div className="flex items-center justify-end gap-1 mt-1">
                  <span className="text-[10px] text-muted-foreground">
                    {format(new Date(message.timestamp), "HH:mm", { locale: tr })}
                  </span>
                  {isOwn && (
                    <span className="text-muted-foreground">
                      {message.status === "read" ? (
                        <CheckCheck className="w-4 h-4 text-primary" />
                      ) : message.status === "delivered" ? (
                        <CheckCheck className="w-4 h-4" />
                      ) : (
                        <Check className="w-4 h-4" />
                      )}
                    </span>
                  )}
                </div>
              </div>
            </div>
          )
        })}
        <div ref={messagesEndRef} />
      </div>

      <div className="p-3 bg-secondary relative z-10">
        <div className="flex items-center gap-2">
          <EmojiPicker onEmojiSelect={handleEmojiSelect} />
          <Button
            variant="ghost"
            size="icon"
            className="text-muted-foreground hover:text-foreground hover:bg-popover shrink-0"
          >
            <Paperclip className="w-6 h-6" />
          </Button>
          <Input
            placeholder="Mesaj yazın"
            value={newMessage}
            onChange={(e) => setNewMessage(e.target.value)}
            onKeyDown={handleKeyPress}
            className="flex-1 bg-card border-0 text-foreground placeholder:text-muted-foreground focus-visible:ring-0"
          />
          {newMessage.trim() ? (
            <Button
              onClick={handleSend}
              size="icon"
              className="bg-primary hover:bg-primary/90 text-primary-foreground shrink-0"
            >
              <Send className="w-5 h-5" />
            </Button>
          ) : (
            <Button
              variant="ghost"
              size="icon"
              className="text-muted-foreground hover:text-foreground hover:bg-popover shrink-0"
            >
              <Mic className="w-6 h-6" />
            </Button>
          )}
        </div>
      </div>
    </div>
  )
}
