"use client"

import { useState } from "react"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { X, Search, ChevronUp, ChevronDown } from "lucide-react"
import type { Message } from "@/lib/types"

interface MessageSearchProps {
  messages: Message[]
  onClose: () => void
  onNavigate: (messageId: string) => void
}

export function MessageSearch({ messages, onClose, onNavigate }: MessageSearchProps) {
  const [query, setQuery] = useState("")
  const [currentIndex, setCurrentIndex] = useState(0)

  const results = query.trim() ? messages.filter((m) => m.content.toLowerCase().includes(query.toLowerCase())) : []

  const handleNext = () => {
    if (results.length > 0) {
      const newIndex = (currentIndex + 1) % results.length
      setCurrentIndex(newIndex)
      onNavigate(results[newIndex].id)
    }
  }

  const handlePrev = () => {
    if (results.length > 0) {
      const newIndex = currentIndex === 0 ? results.length - 1 : currentIndex - 1
      setCurrentIndex(newIndex)
      onNavigate(results[newIndex].id)
    }
  }

  const handleSearch = (value: string) => {
    setQuery(value)
    setCurrentIndex(0)
    if (value.trim()) {
      const filtered = messages.filter((m) => m.content.toLowerCase().includes(value.toLowerCase()))
      if (filtered.length > 0) {
        onNavigate(filtered[0].id)
      }
    }
  }

  return (
    <div className="h-14 px-4 flex items-center gap-2 bg-secondary border-b border-border">
      <div className="relative flex-1">
        <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
        <Input
          autoFocus
          placeholder="Mesajlarda ara..."
          value={query}
          onChange={(e) => handleSearch(e.target.value)}
          className="pl-10 bg-card border-0 text-foreground placeholder:text-muted-foreground focus-visible:ring-0"
        />
      </div>
      {results.length > 0 && (
        <span className="text-sm text-muted-foreground whitespace-nowrap">
          {currentIndex + 1} / {results.length}
        </span>
      )}
      <div className="flex items-center gap-1">
        <Button
          variant="ghost"
          size="icon"
          onClick={handlePrev}
          disabled={results.length === 0}
          className="text-muted-foreground hover:text-foreground hover:bg-popover"
        >
          <ChevronUp className="w-5 h-5" />
        </Button>
        <Button
          variant="ghost"
          size="icon"
          onClick={handleNext}
          disabled={results.length === 0}
          className="text-muted-foreground hover:text-foreground hover:bg-popover"
        >
          <ChevronDown className="w-5 h-5" />
        </Button>
      </div>
      <Button
        variant="ghost"
        size="icon"
        onClick={onClose}
        className="text-muted-foreground hover:text-foreground hover:bg-popover"
      >
        <X className="w-5 h-5" />
      </Button>
    </div>
  )
}
