"use client"

import { useState } from "react"
import { useChat } from "@/lib/chat-context"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { ArrowLeft, Search } from "lucide-react"

interface ContactListProps {
  onBack: () => void
}

export function ContactList({ onBack }: ContactListProps) {
  const { contacts, startChat } = useChat()
  const [searchQuery, setSearchQuery] = useState("")

  const filteredContacts = contacts.filter((contact) =>
    contact.displayName.toLowerCase().includes(searchQuery.toLowerCase()),
  )

  const handleContactClick = (contactId: string) => {
    startChat(contactId)
    onBack()
  }

  return (
    <div className="w-full md:w-[400px] h-full flex flex-col bg-card border-r border-border">
      <div className="h-16 px-4 flex items-center gap-4 bg-secondary">
        <Button variant="ghost" size="icon" onClick={onBack} className="text-foreground hover:bg-popover">
          <ArrowLeft className="w-5 h-5" />
        </Button>
        <span className="font-semibold text-foreground">Yeni Sohbet</span>
      </div>

      <div className="p-2 bg-card">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
          <Input
            placeholder="Kişi ara"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-10 bg-secondary border-0 text-foreground placeholder:text-muted-foreground focus-visible:ring-0"
          />
        </div>
      </div>

      <div className="flex-1 overflow-y-auto">
        {filteredContacts.map((contact) => (
          <div
            key={contact.id}
            onClick={() => handleContactClick(contact.id)}
            className="flex items-center gap-3 px-4 py-3 cursor-pointer hover:bg-secondary transition-colors"
          >
            <Avatar className="w-12 h-12">
              <AvatarFallback className="bg-accent text-foreground">
                {contact.displayName?.charAt(0).toUpperCase()}
              </AvatarFallback>
            </Avatar>
            <div className="flex-1 min-w-0">
              <span className="font-medium text-foreground block truncate">{contact.displayName}</span>
              <span className="text-sm text-muted-foreground truncate block">{contact.status}</span>
            </div>
          </div>
        ))}
      </div>
    </div>
  )
}
